/*
 * pgpPubKey.h -- Structures for PGP Public/Private Keys
 *
 * Written by:	Derek Atkins <warlord@MIT.EDU>
 *
 * $Id: pgpPubKey.h,v 1.14 2001/03/29 00:49:10 hal Exp $
 */

#ifndef Included_pgpPubKey_h
#define Included_pgpPubKey_h

#include "pgpUsuals.h"

PGP_BEGIN_C_DECLARATIONS

#include "pgpOpaqueStructs.h"
#include "pgpPublicKey.h"
#include "pgpStr2Key.h"


struct PGPPubKey {
	PGPContextRef	context;
	
	PGPPubKey *next;
	PGPByte pkAlg;
	PGPByte keyID[8];
	PGPKeyDBObjRef keyDBObj;
	PGPBoolean token;
	void *priv;
	void (*destroy) (PGPPubKey *pubkey);
	/* The sizes of buffers required for encrypt, etc. */
	PGPSize (*maxesk) (PGPPubKey const *pubkey,
					PGPPublicKeyMessageFormat format);
	PGPSize (*maxdecrypted) (PGPPubKey const *pubkey,
					PGPPublicKeyMessageFormat format);
	PGPSize (*maxsig) (PGPPubKey const *pubkey,
					PGPPublicKeyMessageFormat format);
	int (*encrypt) (PGPPubKey const *pubkey, PGPByte const *key,
					PGPSize keylen, PGPByte *esk, PGPSize *esklen,
					PGPRandomContext const *rc,
					PGPPublicKeyMessageFormat format);
	int (*verify) (PGPPubKey const *pubkey,
					PGPByte const *sig, PGPSize siglen,
					PGPHashVTBL const *h, PGPByte const *hash,
					PGPPublicKeyMessageFormat format);
	PGPSize (*bufferLength)(PGPPubKey const *pubkey);
	void (*toBuffer)(PGPPubKey const *pubkey, PGPByte *buf);
	int (*pubparams) (PGPPubKey const *pubkey, BigNum *p1, BigNum *p2,
					  BigNum *p3, BigNum *p4, BigNum *p5, BigNum *p6,
					  BigNum *p7, BigNum *p8);
	void (*setkeyid)(PGPPubKey *pubkey, PGPByte *keyid);
};


struct PGPSecKey {
	PGPContextRef	context;
	
	PGPByte pkAlg;
	PGPByte keyID[8];
	PGPKeyDBObjRef keyDBObj;
	PGPBoolean token;
	void *priv;
	void (*destroy) (PGPSecKey *seckey);
	PGPPubKey * (*pubkey) (PGPSecKey const *seckey);
	int (*islocked) (PGPSecKey const *seckey);
	PGPError (*lockingalgorithm) (PGPSecKey const *seckey,
					PGPCipherAlgorithm *alg, PGPSize *algKeySize);
	PGPError (*s2ktype) (PGPSecKey const *seckey,
					PGPStringToKeyType *s2kType);
	PGPError (*convertpassphrase) (PGPSecKey *seckey, PGPEnv const *env,
					char const *phrase, PGPSize plen, PGPByte *outbuf);
	int (*unlock) (PGPSecKey *seckey,
					char const *phrase, PGPSize plen, PGPBoolean hashedPhrase);
	void (*lock) (PGPSecKey *seckey);
	/* The sizes of buffers required for decrypt, etc. */
	PGPSize (*maxesk) (PGPSecKey const *seckey,
					PGPPublicKeyMessageFormat format);
	PGPSize (*maxdecrypted) (PGPSecKey const *seckey,
					PGPPublicKeyMessageFormat format);
	PGPSize (*maxsig) (PGPSecKey const *seckey,
					PGPPublicKeyMessageFormat format);
	int (*decrypt) (PGPSecKey *seckey,
					PGPByte const *esk, PGPSize esklen,
					PGPByte *key, PGPSize *keylen,
					char const *phrase, PGPSize plen,
					PGPPublicKeyMessageFormat format);
	int (*sign) (PGPSecKey *seckey,
					PGPHashVTBL const *h, PGPByte const *hash,
					PGPByte *sig, PGPSize *siglen,
					PGPRandomContext const *rc,
					PGPPublicKeyMessageFormat format);
	int (*changeLock)(PGPSecKey *seckey, PGPEnv const *env,
					PGPRandomContext const *rc,
					char const *ophrase, PGPSize oplen,
					PGPBoolean oHashedPhrase,
					char const *phrase, PGPSize plen,
					PGPStringToKeyType s2ktype);
	PGPSize (*bufferLength)(PGPSecKey const *seckey);
	void (*toBuffer)(PGPSecKey const *seckey, PGPByte *buf);
	int (*secparams) (PGPSecKey const *pubkey, BigNum *p1, BigNum *p2,
					  BigNum *p3, BigNum *p4, BigNum *p5, BigNum *p6,
					  BigNum *p7, BigNum *p8);
	void (*setkeyid)(PGPSecKey *pubkey, PGPByte *keyid);
};


#include "pgpKeys.h"	/* to get 'PGPPublicKeyAlgorithm' type */
/*
 * Note on kPGPPublicKeyAlgorithm_RSAEncryptOnly/SignOnly:
 *
 * These are ViaCrypt's "restricted" versions of RSA.  There are reasons
 * to want PGP to limit you in this way.  Some forces which might try
 * to force disclosure of your key (such as courts) can be dissuaded on
 * the grounds that nothing is being hidden by the keys.
 *
 * The *annoying* thing, however, is that ViaCrypt chose to leave the
 * encrypted session kay and signature packets with a pkalg byte of 1.
 * Which means that various bits of code contain kludges to deal with
 * this fact.
 */

#define PGP_PKUSE_SIGN          0x01
#define PGP_PKUSE_ENCRYPT       0x02
#define PGP_PKUSE_SIGN_ENCRYPT  (PGP_PKUSE_SIGN | PGP_PKUSE_ENCRYPT)

#define pgpPubKeyNext(p)	(p)->next
#define pgpPubKeyDestroy(p)	(p)->destroy(p)
#define pgpPubKeyMaxesk(p,f)	(p)->maxesk(p,f)
#define pgpPubKeyMaxdecrypted(p,f)	(p)->maxdecrypted(p,f)
#define pgpPubKeyMaxsig(p,f)	(p)->maxsig(p,f)
#define pgpPubKeyMaxsighash(p,f)	(p)->maxsighash(p,f)
#define pgpPubKeyEncrypt(p,k,kl,e,el,r,f)	(p)->encrypt(p,k,kl,e,el,r,f)
#define pgpPubKeyVerify(p,s,sl,h,ha,f)	(p)->verify(p,s,sl,h,ha,f)
#define pgpPubKeyBufferLength(p)	(p)->bufferLength(p)
#define pgpPubKeyToBuffer(p,b)		(p)->toBuffer(p,b)
#define pgpPubKeyParams(p,p1,p2,p3,p4,p5,p6,p7,p8)  (p)->pubparams(p,p1,p2,p3,p4,p5,p6,p7,p8)
#define pgpPubKeySetKeyID(p,k)	(p)->setkeyid(p,k)
#define pgpPubKeyOnToken(p)	((p)->token != 0)

#define pgpSecKeyDestroy(s)	(s)->destroy(s)
#define pgpSecKeyPubkey(s)	(s)->pubkey(s)
#define pgpSecKeyIslocked(s)	(s)->islocked(s)
#define pgpSecKeyLockingalgorithm(s,a,k)	(s)->lockingalgorithm(s,a,k)
#define pgpSecKeyS2Ktype(s,t)	(s)->s2ktype(s,t)
#define pgpSecKeyConvertPassphrase(s,e,p,l,o) \
					(s)->convertpassphrase(s,e,p,l,o)
#define pgpSecKeyIslocked(s)	(s)->islocked(s)
#define pgpSecKeyUnlock(s,p,pl,hp)	(s)->unlock(s,p,pl,hp)
#define pgpSecKeyLock(s)	(s)->lock(s)
#define pgpSecKeyMaxesk(s,f)	(s)->maxesk(s,f)
#define pgpSecKeyMaxdecrypted(s,f)	(s)->maxdecrypted(s,f)
#define pgpSecKeyMaxsig(s,f)	(s)->maxsig(s,f)
#define pgpSecKeyMaxsighash(s,f)	(s)->maxsighash(s,f)
#define pgpSecKeyDecrypt(s,e,el,k,kl,p,pl,f)  (s)->decrypt(s,e,el,k,kl,p,pl,f)
#define pgpSecKeySign(s,h,ha,si,sil,r,f)	(s)->sign(s,h,ha,si,sil,r,f)
#define pgpSecKeyChangeLock(s,e,r,op,opl,hp,p,pl,t) (s)->changeLock(s,e,r,op,opl,hp,p,pl,t)
#define pgpSecKeyBufferLength(s)	(s)->bufferLength(s)
#define pgpSecKeyToBuffer(s,b)		(s)->toBuffer(s,b)
#define pgpSecKeyParams(s,p1,p2,p3,p4,p5,p6,p7,p8)  (s)->secparams(s,p1,p2,p3,p4,p5,p6,p7,p8)
#define pgpSecKeySetKeyID(s,k)	(s)->setkeyid(s,k)
#define pgpSecKeyOnToken(s)	((s)->token != 0)

PGPPkAlg const  *pgpPkalgByNumber(PGPByte pkalg);
PGPUInt32  pgpKeyAlgUse(PGPPkAlg const *pkAlg); 
PGPSize  pgpPubKeyPrefixSize(PGPByte pkAlg, PGPByte const *p, PGPSize len);

PGPPubKey  *	pgpPubKeyFromBuf(PGPContextRef	context,
					PGPByte pkAlg, PGPByte const *p, PGPSize len,
					PGPError *error);
					
PGPSecKey  *	pgpSecKeyFromBuf( PGPContextRef	context,
					PGPByte pkAlg, PGPByte const *p, PGPSize len,
					PGPBoolean v3, void *tok, PGPError *error);

unsigned			pgpSecKeyEntropy(PGPPkAlg const *pkAlg,
						unsigned bits, PGPBoolean fastgen, PGPBoolean v3,
						void *tokenobj);
						
PGPSecKey  *	pgpSecKeyGenerate(PGPContextRef	context,
					PGPPkAlg const *pkAlg,
					unsigned bits, PGPBoolean fastgen, PGPBoolean v3,
					PGPRandomContext const *rc,
					int (*progress)(void *arg, int c),
					void *arg, void *tokenobj,
					PGPByte const *passphrase, PGPSize passphraseLength,
					PGPBoolean genMaster, PGPError *error);

PGPBoolean		pgpIsKeyRelatedError( PGPError err );


PGP_END_C_DECLARATIONS

#endif /* Included_pgpPubKey_h */
